#include <limits>

#include "gtest/gtest.h"
#include "src/complexNumbers.hpp"

TEST(ComplexNumberTest, TestComplexMagnitude) {
  // Arrange
  ComplexNumber a(3.0, 4.0);
  
  // Act
  auto magnitude = a.magnitude();
  
  // Assert
  ASSERT_DOUBLE_EQ(magnitude, 5.0);
}

TEST(ComplexNumberTest, TestComplexConjugate) {
  // Arrange
  ComplexNumber a(3.0, 4.0);
  
  // Act
  a.conjugate();
  
  // Assert
  ASSERT_DOUBLE_EQ(a.Re(), 3.0);
  ASSERT_DOUBLE_EQ(a.Im(), -4.0);
}

TEST(ComplexNumberTest, TestComplexNumberWithNaN) {
  // Arrange
  ComplexNumber a(1.2, -0.3);
  ComplexNumber b(1.8, 5.3);
  
  // Act
  a.setRe(std::numeric_limits<double>::quiet_NaN());
  
  // Assert
  ASSERT_THROW(ComplexNumber a(1.2, std::numeric_limits<double>::quiet_NaN()), std::runtime_error);
  ASSERT_THROW(a + b, std::runtime_error);
  ASSERT_THROW(a - b, std::runtime_error);
  ASSERT_THROW(a * b, std::runtime_error);
  ASSERT_THROW(a / b, std::runtime_error);
}

TEST(ComplexNumberTest, TestComplexAddition) {
  // Arrange
  ComplexNumber a(1.2, -0.3);
  ComplexNumber b(1.8, 5.3);
  
  // Act
  ComplexNumber c = a + b;
  
  // Assert
  ASSERT_DOUBLE_EQ(c.Re(), 3.0);
  ASSERT_DOUBLE_EQ(c.Im(), 5.0);
}

TEST(ComplexNumberTest, TestComplexSubtraction) {
  // Arrange
  ComplexNumber a(1.2, -0.3);
  ComplexNumber b(1.8, 5.3);
  
  // Act
  ComplexNumber c = a - b;
  
  // Assert
  ASSERT_DOUBLE_EQ(c.Re(), -0.6);
  ASSERT_DOUBLE_EQ(c.Im(), -5.6);
}

TEST(ComplexNumberTest, TestComplexMultiplication) {
  // Arrange
  ComplexNumber a(1.2, -0.3);
  ComplexNumber b(1.8, 5.3);
  
  // Act
  ComplexNumber c = a * b;
  
  // Assert
  ASSERT_DOUBLE_EQ(c.Re(), 3.75);
  ASSERT_DOUBLE_EQ(c.Im(), 5.82);
}

TEST(ComplexNumberTest, TestComplexDivision) {
  // Arrange
  ComplexNumber a(1.0, -2.0);
  ComplexNumber b(1.0, 2.0);
  
  // Act
  ComplexNumber c = a / b;
  
  // Assert
  ASSERT_DOUBLE_EQ(c.Re(), -0.6);
  ASSERT_DOUBLE_EQ(c.Im(), -0.8);
}

TEST(ComplexNumberTest, TestComplexDivisionWithNaN) {
  // Arrange
  ComplexNumber a(1.0, -2.0);
  ComplexNumber b(0.0, 0.0);
  
  // Act
  
  // Assert
  ASSERT_THROW(a / b, std::runtime_error);
}

int main(int argc, char **argv) {
  testing::InitGoogleTest(&argc, argv);
  return RUN_ALL_TESTS();
}