#include <filesystem>
#include <vector>

#include "gtest/gtest.h"

#include "meshReaderLib/meshReader.hpp"

class ReadUnstructuredMeshTest : public ::testing::Test {
public:
  ReadUnstructuredMeshTest() {
    std::filesystem::path meshFileWithFamily("mesh/unstructured2D.cgns");
    std::filesystem::path meshFileWithNoFamily("mesh/unstructured2DNoFamily.cgns");

    ReadUnstructuredMesh meshWithFamily(meshFileWithFamily);
    ReadUnstructuredMesh meshWithNoFamily(meshFileWithNoFamily);

    meshWithFamily.readMesh();
    meshWithNoFamily.readMesh();

    unstructuredMesh.push_back(meshWithFamily);
    unstructuredMesh.push_back(meshWithNoFamily);
  }

protected:
  std::vector<ReadUnstructuredMesh> unstructuredMesh;
  std::vector<std::vector<double>> coordinateX {
    {0, 0.5, 1, 1, 1, 0.5, 0, 0, 0.75, 0.5, 0.75, 0.25, 0.25},
    {1, 1.5, 2, 2, 2, 1.5, 1, 1, 1.5}
  };
  std::vector<std::vector<double>> coordinateY {
    {0, 0, 0, 0.5, 1, 1, 1, 0.5, 0.75, 0.5, 0.25, 0.25, 0.75},
    {0, 0, 0, 0.5, 1, 1, 1, 0.5, 0.5}
  };
  std::vector<std::vector<std::vector<unsigned>>> testCells {
    {
      {
        {2, 10, 1},
        {4, 8, 3},
        {12, 5, 6},
        {11, 7, 0},
        {2, 3, 10},
        {12, 6, 7},
        {4, 5, 8},
        {11, 0, 1},
        {8, 5, 12, 9},
        {9, 12, 7, 11},
        {9, 11, 1, 10},
        {8, 9, 10, 3}
      }
    },
    {
      {
        {5, 6, 7, 8},
        {8, 7, 0, 1},
        {8, 1, 2, 3},
        {5, 8, 3, 4}
      }
    }
  };
  std::vector<std::vector<std::vector<std::vector<std::vector<unsigned>>>>> testInterfaces {
    {
      {
        {
          {
            {2, 3},
            {3, 4}
          },
          {
            {0, 7},
            {7, 6}
          }
        }
      }
    },
    {
      {
        {
          {
            {2, 3},
            {3, 4}
          },
          {
            {0, 7},
            {7, 6}
          }
        }
      }
    }
  };
  std::vector<std::vector<std::vector<std::vector<unsigned>>>> testBoundaryConditions {
    {
      {
        {6,7},
        {7,0}
      },
      {
        {4,5},
        {5,6}
      },
      {
        {0,1},
        {1,2}
      }
    },
    {
      {
        {2,3},
        {3,4}
      },
      {
        {4,5},
        {5,6}
      },
      {
        {0,1},
        {1,2}
      }
    }
  };
};

TEST_F(ReadUnstructuredMeshTest, readCoordinates) {
  for (const auto& mesh : unstructuredMesh) {
    // Arrange

    // Act
    auto coordinates = mesh.getCoordinates();

    // Assert
    ASSERT_EQ(coordinates.size(), 2);
    ASSERT_EQ(coordinates[0].size(), 13);
    ASSERT_EQ(coordinates[1].size(), 9);

    for (int zone = 0; zone < coordinates.size(); ++zone) {
      for (int i = 0; i < coordinates[zone].size(); ++i) {
        ASSERT_DOUBLE_EQ(coordinates[zone][i][COORDINATE::X], coordinateX[zone][i]);
        ASSERT_DOUBLE_EQ(coordinates[zone][i][COORDINATE::Y], coordinateY[zone][i]);
      }
    }
  }
}

TEST_F(ReadUnstructuredMeshTest, readInternalCells) {
  for (const auto& mesh : unstructuredMesh) {
    // Arrange

    // Act
    auto cells = mesh.getInternalCells();

    // Assert
    ASSERT_EQ(cells.size(), 2);
    ASSERT_EQ(cells[0].size(), 12);
    ASSERT_EQ(cells[1].size(), 4);

    for (int zone = 0; zone < cells.size(); ++zone) {
      for (int cell = 0; cell < cells[zone].size(); ++cell) {
        for (int vertex = 0; vertex < cells[zone][cell].size(); ++vertex) {
          ASSERT_EQ(cells[zone][cell][vertex], testCells[zone][cell][vertex]);
        }
      }
    }
  }
}

TEST_F(ReadUnstructuredMeshTest, readInterfaces) {
  for (const auto& mesh : unstructuredMesh) {
    // Arrange

    // Act
    auto meshInterfaces = mesh.getInterfaceConnectivity();

    // Assert
    ASSERT_EQ(meshInterfaces.size(), 2);
    ASSERT_EQ(meshInterfaces[0].size(), 1);
    ASSERT_EQ(meshInterfaces[1].size(), 1);

    ASSERT_EQ(meshInterfaces[0][0].zones[0], 0);
    ASSERT_EQ(meshInterfaces[0][0].zones[1], 1);

    ASSERT_EQ(meshInterfaces[1][0].zones[0], 0);
    ASSERT_EQ(meshInterfaces[1][0].zones[1], 1);

    for (int zone = 0; zone < meshInterfaces.size(); ++zone) {
      for (int interface = 0; interface < meshInterfaces[zone].size(); ++interface) {
        ASSERT_EQ(meshInterfaces[zone][interface].ownerIndex.size(), 2);
        ASSERT_EQ(meshInterfaces[zone][interface].neighbourIndex.size(), 2);
        for (int index = 0; index < meshInterfaces[zone][interface].ownerIndex.size(); ++index) {
          auto receivedOwnerStart = meshInterfaces[zone][interface].ownerIndex[index][0];
          auto expectedOwnerStart = testInterfaces[zone][interface][0][index][0];
          ASSERT_EQ(receivedOwnerStart, expectedOwnerStart);

          auto receivedOwnerEnd = meshInterfaces[zone][interface].ownerIndex[index][1];
          auto expectedOwnerEnd = testInterfaces[zone][interface][0][index][1];
          ASSERT_EQ(receivedOwnerEnd, expectedOwnerEnd);

          auto receivedNeighbourStart = meshInterfaces[zone][interface].neighbourIndex[index][0];
          auto expectedNeighbourStart = testInterfaces[zone][interface][1][index][0];
          ASSERT_EQ(receivedNeighbourStart, expectedNeighbourStart);

          auto receivedNeighbourEnd = meshInterfaces[zone][interface].neighbourIndex[index][1];
          auto expectedNeighbourEnd = testInterfaces[zone][interface][1][index][1];
          ASSERT_EQ(receivedNeighbourEnd, expectedNeighbourEnd);
        }
      }
    }
  }
}

TEST_F(ReadUnstructuredMeshTest, readBoundaryConditions) {
  for (const auto& mesh : unstructuredMesh) {
    // Arrange

    // Act
    auto bc = mesh.getBoundaryConditions();

    // Assert
    ASSERT_EQ(bc.size(), 2);

    ASSERT_EQ(bc[0][0].boundaryType, BC::INLET);
    ASSERT_EQ(bc[0][1].boundaryType, BC::SYMMETRY);
    ASSERT_EQ(bc[0][2].boundaryType, BC::WALL);

    ASSERT_EQ(bc[1][0].boundaryType, BC::OUTLET);
    ASSERT_EQ(bc[1][1].boundaryType, BC::SYMMETRY);
    ASSERT_EQ(bc[1][2].boundaryType, BC::WALL);

    for (int zone = 0; zone < bc.size(); ++zone) {
      ASSERT_EQ(bc[zone].size(), 3);
      for (int boundary = 0; boundary < bc[zone].size(); ++boundary) {
        ASSERT_EQ(bc[zone][boundary].indices.size(), 2);
        for (int index = 0; index < bc[zone][boundary].indices.size(); ++index) {
          auto receivedStart = bc[zone][boundary].indices[index][0];
          auto expectedStart = testBoundaryConditions[zone][boundary][index][0];
          ASSERT_EQ(receivedStart, expectedStart);

          auto receivedEnd = bc[zone][boundary].indices[index][1];
          auto expectedEnd = testBoundaryConditions[zone][boundary][index][1];
          ASSERT_EQ(receivedEnd, expectedEnd);
        }
      }
    }
  }
}