#include <filesystem>
#include <vector>

#include "gtest/gtest.h"

#include "meshReaderLib/meshReader.hpp"

class ReadStructuredMeshTest : public ::testing::Test {
public:
  ReadStructuredMeshTest() {
    std::filesystem::path meshFileWithFamily("mesh/structured2D.cgns");
    std::filesystem::path meshFileWithNoFamily("mesh/structured2DNoFamily.cgns");

    ReadStructuredMesh meshWithFamily(meshFileWithFamily);
    ReadStructuredMesh meshWithNoFamily(meshFileWithNoFamily);

    meshWithFamily.readMesh();
    meshWithNoFamily.readMesh();

    structuredMesh.push_back(meshWithFamily);
    structuredMesh.push_back(meshWithNoFamily);
  }

protected:
  std::vector<ReadStructuredMesh> structuredMesh;
  std::vector<std::vector<std::vector<double>>> testCoordinateX {
    {
      {0.00, 0.00, 0.00, 0.00, 0.00},
      {0.25, 0.25, 0.25, 0.25, 0.25},
      {0.50, 0.50, 0.50, 0.50, 0.50},
      {0.75, 0.75, 0.75, 0.75, 0.75},
      {1.00, 1.00, 1.00, 1.00, 1.00}
    },
    {
      {1.00, 1.00, 1.00, 1.00, 1.00},
      {1.25, 1.25, 1.25, 1.25, 1.25},
      {1.50, 1.50, 1.50, 1.50, 1.50},
      {1.75, 1.75, 1.75, 1.75, 1.75},
      {2.00, 2.00, 2.00, 2.00, 2.00}
    }
  };
  std::vector<std::vector<std::vector<double>>> testCoordinateY {
    {
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00}
    },
    {
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00},
      {0.00, 0.25, 0.50, 0.75, 1.00}
    }
  };
  std::vector<std::vector<std::vector<unsigned>>> testInterfaces {
    {
      {4, 4, 4, 4, 4},
      {0, 1, 2, 3, 4}
    },
    {
      {0, 0, 0, 0, 0},
      {0, 1, 2, 3, 4}
    }
  };
  std::vector<std::vector<std::vector<std::vector<unsigned>>>> testBoundaryConditions {
    {
      {
        {0, 1, 2, 3, 4},
        {0, 0, 0, 0, 0}
      },
      {
        {0, 1, 2, 3, 4},
        {4, 4, 4, 4, 4}
      },
      {
        {0, 0, 0, 0, 0},
        {0, 1, 2, 3, 4}
      }
    },
    {
      {
        {0, 1, 2, 3, 4},
        {0, 0, 0, 0, 0}
      },
      {
        {4, 4, 4, 4, 4},
        {0, 1, 2, 3, 4}
      },
      {
        {0, 1, 2, 3, 4},
        {4, 4, 4, 4, 4}
      }
    }
  };
};

TEST_F(ReadStructuredMeshTest, readCoordinates) {
  for (const auto& mesh : structuredMesh) {
    // Arrange

    // Act
    auto coordinates = mesh.getCoordinates();

    // Assert
    ASSERT_EQ(coordinates.size(), 2);
    for (int zone = 0; zone < coordinates.size(); ++zone) {
      ASSERT_EQ(coordinates[zone].size(), 5);
      for (int i = 0; i < coordinates[zone].size(); ++i) {
        ASSERT_EQ(coordinates[zone][i].size(), 5);
        for (int j = 0; j < coordinates[zone][i].size(); ++j) {
          ASSERT_DOUBLE_EQ(coordinates[zone][i][j][COORDINATE::X], testCoordinateX[zone][i][j]);
          ASSERT_DOUBLE_EQ(coordinates[zone][i][j][COORDINATE::Y], testCoordinateY[zone][i][j]);
        }
      }
    }
  }
}

TEST_F(ReadStructuredMeshTest, readInterfaces) {
  for (const auto& mesh : structuredMesh) {
    // Arrange

    // Act
    auto meshInterfaces = mesh.getInterfaceConnectivity();

    // Assert
    ASSERT_EQ(meshInterfaces.size(), 1);
    ASSERT_EQ(meshInterfaces[0].zones[0], 0);
    ASSERT_EQ(meshInterfaces[0].zones[1], 1);
    ASSERT_EQ(meshInterfaces[0].ownerIndex.size(), 5);
    ASSERT_EQ(meshInterfaces[0].neighbourIndex.size(), 5);

    for (int interface = 0; interface < meshInterfaces.size(); ++interface) {
      for (int index = 0; index < meshInterfaces[interface].ownerIndex.size(); ++index) {
        auto ownerSize = meshInterfaces[interface].ownerIndex.size();
        auto neighbourSize = meshInterfaces[interface].neighbourIndex.size();
        ASSERT_EQ(ownerSize, neighbourSize);

        auto receivedOwnerX = meshInterfaces[interface].ownerIndex[index][COORDINATE::X];
        auto expectedOwnerX = testInterfaces[0][COORDINATE::X][index];
        ASSERT_EQ(receivedOwnerX, expectedOwnerX);

        auto receivedOwnerY = meshInterfaces[interface].ownerIndex[index][COORDINATE::Y];
        auto expectedOwnerY = testInterfaces[0][COORDINATE::Y][index];
        ASSERT_EQ(receivedOwnerY, expectedOwnerY);

        auto receivedNeighbourX = meshInterfaces[interface].neighbourIndex[index][COORDINATE::X];
        auto expectedNeighbourX = testInterfaces[1][COORDINATE::X][index];
        ASSERT_EQ(receivedNeighbourX, expectedNeighbourX);

        auto receivedNeighbourY = meshInterfaces[interface].neighbourIndex[index][COORDINATE::Y];
        auto expectedNeighbourY = testInterfaces[1][COORDINATE::Y][index];
        ASSERT_EQ(receivedNeighbourY, expectedNeighbourY);
      }
    }
  }
}

TEST_F(ReadStructuredMeshTest, readBoundaryConditions) {
  for (const auto& mesh : structuredMesh) {
    // Arrange

    // Act
    auto bc = mesh.getBoundaryConditions();

    // Assert
    ASSERT_EQ(bc.size(), 2);
    ASSERT_EQ(bc[0].size(), 3);
    ASSERT_EQ(bc[1].size(), 3);

    ASSERT_EQ(bc[0][0].boundaryType, BC::WALL);
    ASSERT_EQ(bc[0][1].boundaryType, BC::SYMMETRY);
    ASSERT_EQ(bc[0][2].boundaryType, BC::INLET);

    ASSERT_EQ(bc[1][0].boundaryType, BC::WALL);
    ASSERT_EQ(bc[1][1].boundaryType, BC::OUTLET);
    ASSERT_EQ(bc[1][2].boundaryType, BC::SYMMETRY);

    for (int zone = 0; zone < bc.size(); ++zone) {
      for (int boundary = 0; boundary < bc[zone].size(); ++boundary) {
        ASSERT_EQ(bc[zone][boundary].indices.size(), 5);
        for (int vertex = 0; vertex < bc[zone][boundary].indices.size(); ++vertex) {
          auto receivedValueX = bc[zone][boundary].indices[vertex][COORDINATE::X];
          auto expectedValueX = testBoundaryConditions[zone][boundary][COORDINATE::X][vertex];
          ASSERT_EQ(receivedValueX, expectedValueX);

          auto receivedValueY = bc[zone][boundary].indices[vertex][COORDINATE::Y];
          auto expectedValueY = testBoundaryConditions[zone][boundary][COORDINATE::Y][vertex];
          ASSERT_EQ(receivedValueY, expectedValueY);
        }
      }
    }
  }
}