#include "linearAlgebraLib/src/vector.hpp"

namespace linearAlgebraLib {

Vector::Vector(const unsigned &size) {
  _vector.resize(size, 0.0);
}

unsigned Vector::size() const {
  return _vector.size();
}

Vector Vector::transpose() {
  Vector transposedVector(_vector.size());
  transposedVector._vector = _vector;
  transposedVector._isRowVector = _isRowVector == true ? false : true;
  return transposedVector;
}

double Vector::getL2Norm() const {
  double L2Norm = 0.0;
  for (const auto &entry : _vector)
    L2Norm += std::pow(entry, 2);
  L2Norm = std::sqrt(L2Norm);
  return L2Norm;
}

Vector Vector::operator+(const Vector &other) {
  assert(_vector.size() == other._vector.size() && "vectors must have the same dimension");
  Vector resultVector(_vector.size());
  resultVector._vector.resize(_vector.size());
  for (unsigned i = 0; i < _vector.size(); ++i)
    resultVector._vector[i] = _vector[i] + other._vector[i];
  return resultVector;
}

Vector Vector::operator-(const Vector &other) {
  assert(_vector.size() == other._vector.size() && "vectors must have the same dimension");
  Vector resultVector(_vector.size());
  resultVector._vector.resize(_vector.size());
  for (unsigned i = 0; i < _vector.size(); ++i)
    resultVector._vector[i] = _vector[i] - other._vector[i];
  return resultVector;
}

double &Vector::operator[](unsigned index) {
  return _vector[index];
}

const double &Vector::operator[](unsigned index) const {
  return _vector[index];
}

Vector &Vector::operator*(const double &scaleFactor) {
  for (unsigned index = 0; index < _vector.size(); ++index)
    _vector[index] *= scaleFactor;
  return *this;
}

double Vector::operator*(const Vector &other) {
  assert(_isRowVector && !other._isRowVector && "first vector must be a row vector, second must be a column vector");
  double sum = 0.0;
  for (unsigned i = 0; i < _vector.size(); ++i)
    sum += _vector[i] * other._vector[i];
  return sum;
}

Vector operator*(const double &scaleFactor, Vector vector) {
  for (unsigned index = 0; index < vector.size(); ++index)
    vector[index] *= scaleFactor;
  return vector;
}

std::ostream &operator<<(std::ostream &out, const Vector &vector) {
  out << "( ";
  for (unsigned index = 0; index < vector.size(); ++index)
    out << vector[index] << " ";
  out << ")";
  return out;
}

} // namespace linearAlgebraLib