#!/bin/bash

# specify library versions to be installed
GTEST_VERSION=1.14.0

# specify install directory (IMPORTANT: do not remove trailing forward slash, e.g. /)
# you can leave it as it is if you want to install CGNS in your home directory
INSTALL_DIR=~/libs/

###########################
### do not edit lines below
###########################

# exit upon error
set -e

# check that required programs are installed
if ! [ -x "$(command -v wget)" ]; then
  echo 'Error: wget is not installed.' >&2
  echo 'Please install wget first. (e.g. on ubuntu: sudo apt install -y wget)'
  exit 1
fi

if ! [ -x "$(command -v cmake)" ]; then
  echo 'Error: cmake is not installed.' >&2
  echo 'Please install cmake first. (e.g. on ubuntu: sudo apt install -y cmake)'
  exit 1
fi

if ! [ -x "$(command -v ninja)" ]; then
  echo 'Error: ninja is not installed.' >&2
  echo 'Please install ninja first. (e.g. on ubuntu: sudo apt install -y ninja-build)'
  exit 1
fi

if ! [ -x "$(command -v g++)" ]; then
  echo 'Error: g++ is not installed.' >&2
  echo 'Please install g first. (e.g. on ubuntu: sudo apt install -y build-essential)'
  exit 1
fi

# create install directory
mkdir -p $INSTALL_DIR

# construct dynamic download links
GTEST_URL=https://github.com/google/googletest/archive/refs/tags/v$GTEST_VERSION.tar.gz

# check if libraries are already downloaded, if not, use wget to download them
if [ ! -d "googletest-$GTEST_VERSION/" ]; then
  wget $GTEST_URL
  mv v$GTEST_VERSION.tar.gz gtest-$GTEST_VERSION.tar.gz
  tar -zxf gtest-$GTEST_VERSION.tar.gz
fi

# set number of CPU cores
NUM_CPU=$(nproc)

# compile and install zlib
cd googletest-$GTEST_VERSION
mkdir -p build && cd build
cmake -DCMAKE_BUILD_TYPE=Release -DCMAKE_INSTALL_PREFIX=$INSTALL_DIR -G Ninja ..
cmake --build . --target install --config Release -j $NUM_CPU
cd ../../

# test that the CGNS library was installed successfully

SAMPLE_CGNS_PROGRAM='
#include "gtest/gtest.h"

TEST(SampleTest, Test1) {
  EXPECT_EQ(1, 1);
}

int main(int argc, char **argv) {
  testing::InitGoogleTest(&argc, argv);
  return RUN_ALL_TESTS();
}
'
echo "$SAMPLE_CGNS_PROGRAM" > gtestTest.cpp

echo ""
echo "Test gtest installation"
echo ""

g++ -I${INSTALL_DIR}/include -c -o gtestTest.o gtestTest.cpp
g++ -o gtestTest gtestTest.o -L${INSTALL_DIR}/lib -lgtest
./gtestTest

echo ""
echo "If you can see the gtest test runner printing its output (1 successful test), then the installation has worked correctly."
echo ""

# clean up
rm gtestTest gtestTest.o

### output information to screen

echo ""
echo "##########################################"
echo ""
echo "The gtest library has been installed in $INSTALL_DIR"
echo ""
echo "- The static test library is located at ${INSTALL_DIR}lib\gtest.lib"
echo "- The static moch library is located at ${INSTALL_DIR}lib\gmock.lib"
echo "- The header file is located at ${INSTALL_DIR}include\gtest\gtest.h"
echo ""
echo "- A minimal program that tests the CGNS installation has been written to gtestTest.cpp"
echo "  To compile it yourself, use the following commands"
echo ""
echo "  g++ -I${INSTALL_DIR::-1}/include -c -o gtestTest.o gtestTest.cpp"
echo "  g++ -o gtestTest gtestTest.o -L${INSTALL_DIR::-1}/lib -lgtest"
echo ""
echo "- You can now run the gtestTest program to test the installation"
echo "  ./gtestTest"
echo ""
echo "##########################################"
echo ""