# make sure correct powershell is set up
Write-Host "To ensure you can compile 64-bit applications, you'll need to create a new PowerShell shortcut."
Write-Host "Step 1:"
Write-Host "- Go to C:\ProgramData\Microsoft\Windows\Start Menu\Programs\Visual Studio 2022\Visual Studio Tools (change the version number if needed)."
Write-Host "Step 2:"
Write-Host "- Create a copy of Developer PowerShell for VS 2022 (change version number if needed)."
Write-Host "Step 3:"
Write-Host "- Name it Developer PowerShell for VS 2022 - x64"
Write-Host "Step 4:"
Write-Host "- Right click on it and append -DevCmdArguments '-arch=x64' to the target, i.e. you have something like:"
Write-Host "C:\Windows\SysWOW64\WindowsPowerShell\v1.0\powershell.exe -noe -c &{Import-Module C:\Program Files\Microsoft Visual Studio\2022\Community\Common7\Tools\Microsoft.VisualStudio.DevShell.dll; Enter-VsDevShell 1eb5d1ab -DevCmdArguments '-arch=x64'}"
Write-Host "Step 5:"
Write-Host "- Launch this script in the new Developer PowerShell for VS 2022 - x64."

Read-Host -Prompt "Press Enter to continue"

# specify library versions to be installed
$GTEST_VERSION = "1.14.0"

# specify install directory (IMPORTANT: do not remove trailing forward slash, e.g. \)
$INSTALL_DIR = "C:\libraries\"

# exit upon error
$ErrorActionPreference = "Stop"

# check that required programs are installed
if (!(Get-Command "cmake" -ErrorAction SilentlyContinue)) {
    Write-Error "Error: cmake is not installed or found."
    exit 1
}

if (!(Get-Command "cl" -ErrorAction SilentlyContinue)) {
    Write-Error "Error: Visual Studio C++ compiler (cl.exe) is not installed or found."
    exit 1
}

# create install directory
New-Item -ItemType Directory -Force -Path $INSTALL_DIR

# construct dynamic download links
$GTEST_URL = "https://github.com/google/googletest/archive/refs/tags/v$GTEST_VERSION.zip"


# check if libraries are already downloaded, if not, use Invoke-WebRequest to download them
if (!(Test-Path "gtest-$GTEST_VERSION")) {
    Invoke-WebRequest -Uri $GTEST_URL -OutFile "gtest-$GTEST_VERSION.zip"
    Expand-Archive -LiteralPath "gtest-$GTEST_VERSION.zip"
}

# set number of CPU cores
$NUM_CPU = (Get-WmiObject -Class Win32_ComputerSystem).NumberOfLogicalProcessors

# compile and install gtest
Set-Location gtest-$GTEST_VERSION\googletest-$GTEST_VERSION
New-Item -ItemType Directory -Force -Path build
Set-Location build
cmake -DCMAKE_BUILD_TYPE=Release -DCMAKE_INSTALL_PREFIX="$INSTALL_DIR" ..
cmake --build . --target install --config Release -j "$NUM_CPU"
Set-Location ..\..\..

# Test that the gtest library was installed successfully
$SAMPLE_GTEST_PROGRAM = @"
#include "gtest/gtest.h"

TEST(SampleTest, Test1) {
  EXPECT_EQ(1, 1);
}

int main(int argc, char **argv) {
  testing::InitGoogleTest(&argc, argv);
  return RUN_ALL_TESTS();
}
"@

# Write the C++ test program to a file
$SAMPLE_GTEST_PROGRAM | Out-File -FilePath gtestTest.cpp -Encoding ASCII

Write-Host ""
Write-Host "Test gtest installation"
Write-Host ""

cl.exe /nologo /EHsc /I "${INSTALL_DIR}include" /c gtestTest.cpp /Fo"gtestTest.obj"
cl.exe /nologo /EHsc gtestTest.obj /Fe"gtestTest.exe" /link /MACHINE:x64 /LIBPATH:"${INSTALL_DIR}lib" gtest.lib
.\gtestTest.exe

Write-Host ""
Write-Host "If you can see the gtest test runner printing its output (1 successful test), then the installation has worked correctly."
Write-Host ""

# clean up
Remove-Item .\gtestTest.obj
Remove-Item .\gtestTest.exe

### output information to screen

Write-Host ""
Write-Host "##########################################"
Write-Host ""
Write-Host "The gtest library has been installed in $INSTALL_DIR"
Write-Host ""
Write-Host "- The static test library is located at ${INSTALL_DIR}lib\gtest.lib"
Write-Host "- The static moch library is located at ${INSTALL_DIR}lib\gmock.lib"
Write-Host "- The header file is located at ${INSTALL_DIR}include\gtest\gtest.h"
Write-Host ""
Write-Host "- A minimal program that tests the gtest installation has been written to gtestTest.cpp"
Write-Host "  To compile it yourself, use the following commands"
Write-Host ""
Write-Host "  Copy-Item "${INSTALL_DIR}bin\zlib.dll" -Destination ."
Write-Host "  Copy-Item "${INSTALL_DIR}bin\hdf5.dll" -Destination ."
Write-Host "  cl.exe /nologo /EHsc /I `"${INSTALL_DIR}include`" /c gtestTest.cpp /Fo`"gtestTest.obj`""
Write-Host "  cl.exe /nologo /EHsc gtestTest.obj /Fe`"gtestTest.exe`" /link /MACHINE:x64 /LIBPATH:`"${INSTALL_DIR}lib`" gtest.lib"
Write-Host ""
Write-Host "- You can now run the gtestTest program to test the installation"
Write-Host "  .\gtestTest.exe"
Write-Host ""
Write-Host "##########################################"
Write-Host ""
