#include <iostream>
#include <filesystem>
#include <cmath>
#include <string>

#include "meshReaderLib/meshReader.hpp"

void readStructuredMeshCoordinates(ReadStructuredMesh &structuredMesh);
void readStructuredMeshInterfaceConnectivity(ReadStructuredMesh &structuredMesh);
void readStructuredMeshBoundaryConditions(ReadStructuredMesh &structuredMesh);

void readUnstructuredMeshCoordinates(ReadUnstructuredMesh &unstructuredMesh);
void readUnstructuredInternalCells(ReadUnstructuredMesh &unstructuredMesh);
void readUnstructuredMeshInterfaceConnectivity(ReadUnstructuredMesh &unstructuredMesh);
void readUnstructuredMeshBoundaryConditions(ReadUnstructuredMesh &unstructuredMesh);

int main() {
  auto structuredMeshPathFamily = std::filesystem::path("mesh/structured2D.cgns");
  ReadStructuredMesh structuredMeshFamily(structuredMeshPathFamily);
  structuredMeshFamily.readMesh();

  readStructuredMeshCoordinates(structuredMeshFamily);
  readStructuredMeshInterfaceConnectivity(structuredMeshFamily);
  readStructuredMeshBoundaryConditions(structuredMeshFamily);

  std::cout << "Structured mesh read successfully from CGNS file (family-based boundary conditions)" << std::endl;

  // structured mesh reading test, without family-based boundary condition
  auto structuredMeshPathNoFamily = std::filesystem::path("mesh/structured2DNoFamily.cgns");
  ReadStructuredMesh structuredMeshNoFamily(structuredMeshPathNoFamily);
  structuredMeshNoFamily.readMesh();

  readStructuredMeshCoordinates(structuredMeshNoFamily);
  readStructuredMeshInterfaceConnectivity(structuredMeshNoFamily);
  readStructuredMeshBoundaryConditions(structuredMeshNoFamily);

  std::cout << "Structured mesh read successfully from CGNS file (non family-based boundary conditions)" << std::endl;

  // unstructured mesh reading test, with family-based boundary condition
  auto unstructuredMeshPathFamily = std::filesystem::path("mesh/unstructured2D.cgns");
  ReadUnstructuredMesh unstructuredMeshFamily(unstructuredMeshPathFamily);
  unstructuredMeshFamily.readMesh();

  readUnstructuredMeshCoordinates(unstructuredMeshFamily);
  readUnstructuredInternalCells(unstructuredMeshFamily);
  readUnstructuredMeshInterfaceConnectivity(unstructuredMeshFamily);
  readUnstructuredMeshBoundaryConditions(unstructuredMeshFamily);

  std::cout << "Unstructured mesh read successfully from CGNS file (family-based boundary conditions)" << std::endl;

  // unstructured mesh reading test, without family-based boundary condition
  auto unstructuredMeshPathNoFamily = std::filesystem::path("mesh/unstructured2DNoFamily.cgns");
  ReadUnstructuredMesh unstructuredMeshNoFamily(unstructuredMeshPathNoFamily);
  unstructuredMeshNoFamily.readMesh();

  readUnstructuredMeshCoordinates(unstructuredMeshNoFamily);
  readUnstructuredInternalCells(unstructuredMeshNoFamily);
  readUnstructuredMeshInterfaceConnectivity(unstructuredMeshNoFamily);
  readUnstructuredMeshBoundaryConditions(unstructuredMeshNoFamily);

  std::cout << "Unstructured mesh read successfully from CGNS file (non family-based boundary conditions)" << std::endl;

  return 0;
}

void readStructuredMeshCoordinates(ReadStructuredMesh &structuredMesh) {
  // get the coordinates from the structured mesh
  auto coordinates = structuredMesh.getCoordinates();

  // test that coordinates were read correctly
  assert(coordinates.size() == 2 && "Expected 2 zones");

  // test zone 0
  assert(std::fabs(coordinates[0][0][0][COORDINATE::X] - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][1][0][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][2][0][COORDINATE::X] - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][3][0][COORDINATE::X] - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][4][0][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[0][0][1][COORDINATE::X] - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][1][1][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][2][1][COORDINATE::X] - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][3][1][COORDINATE::X] - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][4][1][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[0][0][2][COORDINATE::X] - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][1][2][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][2][2][COORDINATE::X] - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][3][2][COORDINATE::X] - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][4][2][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[0][0][3][COORDINATE::X] - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][1][3][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][2][3][COORDINATE::X] - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][3][3][COORDINATE::X] - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][4][3][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[0][0][4][COORDINATE::X] - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][1][4][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][2][4][COORDINATE::X] - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][3][4][COORDINATE::X] - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][4][4][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[0][0][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][1][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][2][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][3][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][4][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[0][0][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][1][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][2][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][3][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][4][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[0][0][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][1][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][2][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][3][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][4][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[0][0][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][1][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][2][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][3][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][4][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[0][0][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][1][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][2][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][3][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][4][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");

  // test zone 1
  assert(std::fabs(coordinates[1][0][0][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][1][0][COORDINATE::X] - 1.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][2][0][COORDINATE::X] - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][3][0][COORDINATE::X] - 1.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][4][0][COORDINATE::X] - 2.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[1][0][1][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][1][1][COORDINATE::X] - 1.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][2][1][COORDINATE::X] - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][3][1][COORDINATE::X] - 1.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][4][1][COORDINATE::X] - 2.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[1][0][2][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][1][2][COORDINATE::X] - 1.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][2][2][COORDINATE::X] - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][3][2][COORDINATE::X] - 1.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][4][2][COORDINATE::X] - 2.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[1][0][3][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][1][3][COORDINATE::X] - 1.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][2][3][COORDINATE::X] - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][3][3][COORDINATE::X] - 1.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][4][3][COORDINATE::X] - 2.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[1][0][4][COORDINATE::X] - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][1][4][COORDINATE::X] - 1.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][2][4][COORDINATE::X] - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][3][4][COORDINATE::X] - 1.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][4][4][COORDINATE::X] - 2.00) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[1][0][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][1][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][2][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][3][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][4][0][COORDINATE::Y] - 0.00) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[1][0][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][1][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][2][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][3][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][4][1][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[1][0][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][1][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][2][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][3][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][4][2][COORDINATE::Y] - 0.50) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[1][0][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][1][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][2][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][3][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][4][3][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[1][0][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][1][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][2][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][3][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][4][4][COORDINATE::Y] - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
}

void readStructuredMeshInterfaceConnectivity(ReadStructuredMesh &structuredMesh) {
  // get the interface connectivity from the structured mesh
  auto interfaceConnectivity = structuredMesh.getInterfaceConnectivity();

  // test that interface connectivity were read correctly
  assert(interfaceConnectivity.size() == 1 && "Expected a single interface");
  assert(interfaceConnectivity[0].zones[0] == 0 && "Expecting owning zone to be zone 0");
  assert(interfaceConnectivity[0].zones[1] == 1 && "Expecting neighbouring zone to be zone 1");
  assert(interfaceConnectivity[0].ownerIndex.size() == 5 && "Expected 5 owner indices");
  assert(interfaceConnectivity[0].neighbourIndex.size() == 5 && "Expected 5 neighbour indices");

  assert(interfaceConnectivity[0].ownerIndex[0][COORDINATE::X] == 4 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].ownerIndex[1][COORDINATE::X] == 4 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].ownerIndex[2][COORDINATE::X] == 4 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].ownerIndex[3][COORDINATE::X] == 4 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].ownerIndex[4][COORDINATE::X] == 4 && "Unexpected owner index in X");

  assert(interfaceConnectivity[0].ownerIndex[0][COORDINATE::Y] == 0 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].ownerIndex[1][COORDINATE::Y] == 1 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].ownerIndex[2][COORDINATE::Y] == 2 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].ownerIndex[3][COORDINATE::Y] == 3 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].ownerIndex[4][COORDINATE::Y] == 4 && "Unexpected owner index in Y");

  assert(interfaceConnectivity[0].neighbourIndex[0][COORDINATE::X] == 0 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].neighbourIndex[1][COORDINATE::X] == 0 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].neighbourIndex[2][COORDINATE::X] == 0 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].neighbourIndex[3][COORDINATE::X] == 0 && "Unexpected owner index in X");
  assert(interfaceConnectivity[0].neighbourIndex[4][COORDINATE::X] == 0 && "Unexpected owner index in X");

  assert(interfaceConnectivity[0].neighbourIndex[0][COORDINATE::Y] == 0 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].neighbourIndex[1][COORDINATE::Y] == 1 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].neighbourIndex[2][COORDINATE::Y] == 2 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].neighbourIndex[3][COORDINATE::Y] == 3 && "Unexpected owner index in Y");
  assert(interfaceConnectivity[0].neighbourIndex[4][COORDINATE::Y] == 4 && "Unexpected owner index in Y");
}

void readStructuredMeshBoundaryConditions(ReadStructuredMesh &structuredMesh) {
  // get the boundary conditions from the structured mesh
  auto boundaryConditions = structuredMesh.getBoundaryConditions();

  assert(boundaryConditions.size() == 2 && "Expected 2 zones");
  assert(boundaryConditions[0].size() == 3 && "Expected to read 3 boundary conditions for the first zone");
  assert(boundaryConditions[1].size() == 3 && "Expected to read 3 boundary conditions for the second zone");

  assert(boundaryConditions[0][0].boundaryType == BC::WALL && "Expected wall boundary condition");
  assert(boundaryConditions[0][1].boundaryType == BC::SYMMETRY && "Expected symmetry boundary condition");
  assert(boundaryConditions[0][2].boundaryType == BC::INLET && "Expected inlet boundary condition");

  assert(boundaryConditions[1][0].boundaryType == BC::WALL && "Expected wall boundary condition");
  assert(boundaryConditions[1][1].boundaryType == BC::OUTLET && "Expected outlet boundary condition");
  assert(boundaryConditions[1][2].boundaryType == BC::SYMMETRY && "Expected symmetry boundary condition");

  assert(boundaryConditions[0][0].indices[0][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[0][0].indices[1][COORDINATE::X] == 1 && "Unexpected index in X");
  assert(boundaryConditions[0][0].indices[2][COORDINATE::X] == 2 && "Unexpected index in X");
  assert(boundaryConditions[0][0].indices[3][COORDINATE::X] == 3 && "Unexpected index in X");
  assert(boundaryConditions[0][0].indices[4][COORDINATE::X] == 4 && "Unexpected index in X");

  assert(boundaryConditions[0][1].indices[0][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[0][1].indices[1][COORDINATE::X] == 1 && "Unexpected index in X");
  assert(boundaryConditions[0][1].indices[2][COORDINATE::X] == 2 && "Unexpected index in X");
  assert(boundaryConditions[0][1].indices[3][COORDINATE::X] == 3 && "Unexpected index in X");
  assert(boundaryConditions[0][1].indices[4][COORDINATE::X] == 4 && "Unexpected index in X");

  assert(boundaryConditions[0][2].indices[0][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[0][2].indices[1][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[0][2].indices[2][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[0][2].indices[3][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[0][2].indices[4][COORDINATE::X] == 0 && "Unexpected index in X");

  assert(boundaryConditions[0][0].indices[0][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[0][0].indices[1][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[0][0].indices[2][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[0][0].indices[3][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[0][0].indices[4][COORDINATE::Y] == 0 && "Unexpected index in Y");

  assert(boundaryConditions[0][1].indices[0][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[0][1].indices[1][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[0][1].indices[2][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[0][1].indices[3][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[0][1].indices[4][COORDINATE::Y] == 4 && "Unexpected index in Y");

  assert(boundaryConditions[0][2].indices[0][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[0][2].indices[1][COORDINATE::Y] == 1 && "Unexpected index in Y");
  assert(boundaryConditions[0][2].indices[2][COORDINATE::Y] == 2 && "Unexpected index in Y");
  assert(boundaryConditions[0][2].indices[3][COORDINATE::Y] == 3 && "Unexpected index in Y");
  assert(boundaryConditions[0][2].indices[4][COORDINATE::Y] == 4 && "Unexpected index in Y");

  assert(boundaryConditions[1][0].indices[0][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[1][0].indices[1][COORDINATE::X] == 1 && "Unexpected index in X");
  assert(boundaryConditions[1][0].indices[2][COORDINATE::X] == 2 && "Unexpected index in X");
  assert(boundaryConditions[1][0].indices[3][COORDINATE::X] == 3 && "Unexpected index in X");
  assert(boundaryConditions[1][0].indices[4][COORDINATE::X] == 4 && "Unexpected index in X");

  assert(boundaryConditions[1][1].indices[0][COORDINATE::X] == 4 && "Unexpected index in X");
  assert(boundaryConditions[1][1].indices[1][COORDINATE::X] == 4 && "Unexpected index in X");
  assert(boundaryConditions[1][1].indices[2][COORDINATE::X] == 4 && "Unexpected index in X");
  assert(boundaryConditions[1][1].indices[3][COORDINATE::X] == 4 && "Unexpected index in X");
  assert(boundaryConditions[1][1].indices[4][COORDINATE::X] == 4 && "Unexpected index in X");

  assert(boundaryConditions[1][2].indices[0][COORDINATE::X] == 0 && "Unexpected index in X");
  assert(boundaryConditions[1][2].indices[1][COORDINATE::X] == 1 && "Unexpected index in X");
  assert(boundaryConditions[1][2].indices[2][COORDINATE::X] == 2 && "Unexpected index in X");
  assert(boundaryConditions[1][2].indices[3][COORDINATE::X] == 3 && "Unexpected index in X");
  assert(boundaryConditions[1][2].indices[4][COORDINATE::X] == 4 && "Unexpected index in X");

  assert(boundaryConditions[1][0].indices[0][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[1][0].indices[1][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[1][0].indices[2][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[1][0].indices[3][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[1][0].indices[4][COORDINATE::Y] == 0 && "Unexpected index in Y");

  assert(boundaryConditions[1][1].indices[0][COORDINATE::Y] == 0 && "Unexpected index in Y");
  assert(boundaryConditions[1][1].indices[1][COORDINATE::Y] == 1 && "Unexpected index in Y");
  assert(boundaryConditions[1][1].indices[2][COORDINATE::Y] == 2 && "Unexpected index in Y");
  assert(boundaryConditions[1][1].indices[3][COORDINATE::Y] == 3 && "Unexpected index in Y");
  assert(boundaryConditions[1][1].indices[4][COORDINATE::Y] == 4 && "Unexpected index in Y");

  assert(boundaryConditions[1][2].indices[0][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[1][2].indices[1][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[1][2].indices[2][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[1][2].indices[3][COORDINATE::Y] == 4 && "Unexpected index in Y");
  assert(boundaryConditions[1][2].indices[4][COORDINATE::Y] == 4 && "Unexpected index in Y");
}

void readUnstructuredMeshCoordinates(ReadUnstructuredMesh &unstructuredMesh) {
  // get the coordinates from the unstructured mesh
  auto coordinates = unstructuredMesh.getCoordinates();

  // test that coordinates were read correctly
  assert(coordinates.size() == 2 && "Expected 2 zones");
  assert(coordinates[0].size() == 13 && "Expected 13 vertices in zone 0");
  assert(coordinates[1].size() == 9 && "Expected 9 vertices in zone 1");

  assert(std::fabs(coordinates[0][0][COORDINATE::X]  - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][1][COORDINATE::X]  - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][2][COORDINATE::X]  - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][3][COORDINATE::X]  - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][4][COORDINATE::X]  - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][5][COORDINATE::X]  - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][6][COORDINATE::X]  - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][7][COORDINATE::X]  - 0.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][8][COORDINATE::X]  - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][9][COORDINATE::X]  - 0.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][10][COORDINATE::X] - 0.75) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][11][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[0][12][COORDINATE::X] - 0.25) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[0][0][COORDINATE::Y]  - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][1][COORDINATE::Y]  - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][2][COORDINATE::Y]  - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][3][COORDINATE::Y]  - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][4][COORDINATE::Y]  - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][5][COORDINATE::Y]  - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][6][COORDINATE::Y]  - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][7][COORDINATE::Y]  - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][8][COORDINATE::Y]  - 0.75) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][9][COORDINATE::Y]  - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][10][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][11][COORDINATE::Y] - 0.25) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[0][12][COORDINATE::Y] - 0.75) < 0.01 && "Unexpected value for Y-coordinate");

  assert(std::fabs(coordinates[1][0][COORDINATE::X]  - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][1][COORDINATE::X]  - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][2][COORDINATE::X]  - 2.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][3][COORDINATE::X]  - 2.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][4][COORDINATE::X]  - 2.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][5][COORDINATE::X]  - 1.50) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][6][COORDINATE::X]  - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][7][COORDINATE::X]  - 1.00) < 0.01 && "Unexpected value for X-coordinate");
  assert(std::fabs(coordinates[1][8][COORDINATE::X]  - 1.50) < 0.01 && "Unexpected value for X-coordinate");

  assert(std::fabs(coordinates[1][0][COORDINATE::Y]  - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][1][COORDINATE::Y]  - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][2][COORDINATE::Y]  - 0.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][3][COORDINATE::Y]  - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][4][COORDINATE::Y]  - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][5][COORDINATE::Y]  - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][6][COORDINATE::Y]  - 1.00) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][7][COORDINATE::Y]  - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
  assert(std::fabs(coordinates[1][8][COORDINATE::Y]  - 0.50) < 0.01 && "Unexpected value for Y-coordinate");
}

void readUnstructuredInternalCells(ReadUnstructuredMesh &unstructuredMesh) {
  auto cells = unstructuredMesh.getInternalCells();

  assert(cells.size() == 2 && "Expected 2 zones");

  assert(cells[0].size() == 12 && "Expected 12 cells in zone 0");
  assert(cells[1].size() == 4 && "Expected 4 cells in zone 1");

  assert(cells[0][0][0] == 2); assert(cells[0][0][1] == 10); assert(cells[0][0][2] == 1);
  assert(cells[0][1][0] == 4); assert(cells[0][1][1] == 8); assert(cells[0][1][2] == 3);
  assert(cells[0][2][0] == 12); assert(cells[0][2][1] == 5); assert(cells[0][2][2] == 6);
  assert(cells[0][3][0] == 11); assert(cells[0][3][1] == 7); assert(cells[0][3][2] == 0);
  assert(cells[0][4][0] == 2); assert(cells[0][4][1] == 3); assert(cells[0][4][2] == 10);
  assert(cells[0][5][0] == 12); assert(cells[0][5][1] == 6); assert(cells[0][5][2] == 7);
  assert(cells[0][6][0] == 4); assert(cells[0][6][1] == 5); assert(cells[0][6][2] == 8);
  assert(cells[0][7][0] == 11); assert(cells[0][7][1] == 0); assert(cells[0][7][2] == 1);
  assert(cells[0][8][0] == 8); assert(cells[0][8][1] == 5); assert(cells[0][8][2] == 12); assert(cells[0][8][3] == 9);
  assert(cells[0][9][0] == 9); assert(cells[0][9][1] == 12); assert(cells[0][9][2] == 7); assert(cells[0][9][3] == 11);
  assert(cells[0][10][0] == 9); assert(cells[0][10][1] == 11); assert(cells[0][10][2] == 1); assert(cells[0][10][3] == 10);
  assert(cells[0][11][0] == 8); assert(cells[0][11][1] == 9); assert(cells[0][11][2] == 10); assert(cells[0][11][3] == 3);

  assert(cells[1][0][0] == 5); assert(cells[1][0][1] == 6); assert(cells[1][0][2] == 7); assert(cells[1][0][3] == 8);
  assert(cells[1][1][0] == 8); assert(cells[1][1][1] == 7); assert(cells[1][1][2] == 0); assert(cells[1][1][3] == 1);
  assert(cells[1][2][0] == 8); assert(cells[1][2][1] == 1); assert(cells[1][2][2] == 2); assert(cells[1][2][3] == 3);
  assert(cells[1][3][0] == 5); assert(cells[1][3][1] == 8); assert(cells[1][3][2] == 3); assert(cells[1][3][3] == 4);
}

void readUnstructuredMeshInterfaceConnectivity(ReadUnstructuredMesh &unstructuredMesh) {
  // get the interface connectivity from the unstructured mesh
  auto interfaceConnectivity = unstructuredMesh.getInterfaceConnectivity();

  // test that interface connectivity was read correctly
  assert(interfaceConnectivity.size() == 2 && "Expected 2 zones");
  assert(interfaceConnectivity[0].size() == 1 && "Expected 1 interface in zone 0");
  assert(interfaceConnectivity[1].size() == 1 && "Expected 1 interface in zone 1");
  
  assert(interfaceConnectivity[0][0].zones[0] == 0 && "Expecting owning zone to be zone 0");
  assert(interfaceConnectivity[0][0].zones[1] == 1 && "Expecting neighbouring zone to be zone 1");

  assert(interfaceConnectivity[0][0].ownerIndex.size() == 2 && "Expected 2 owner indices");
  assert(interfaceConnectivity[0][0].neighbourIndex.size() == 2 && "Expected 2 neighbour indices");

  assert(interfaceConnectivity[0][0].ownerIndex[0][0] == 2 && "Unexpected vertex index");
  assert(interfaceConnectivity[0][0].ownerIndex[0][1] == 3 && "Unexpected vertex index");
  assert(interfaceConnectivity[0][0].ownerIndex[1][0] == 3 && "Unexpected vertex index");
  assert(interfaceConnectivity[0][0].ownerIndex[1][1] == 4 && "Unexpected vertex index");

  assert(interfaceConnectivity[0][0].neighbourIndex[0][0] == 0 && "Unexpected vertex index");
  assert(interfaceConnectivity[0][0].neighbourIndex[0][1] == 7 && "Unexpected vertex index");
  assert(interfaceConnectivity[0][0].neighbourIndex[1][0] == 7 && "Unexpected vertex index");
  assert(interfaceConnectivity[0][0].neighbourIndex[1][1] == 6 && "Unexpected vertex index");

  assert(interfaceConnectivity[1][0].zones[0] == 0 && "Expecting owning zone to be zone 0");
  assert(interfaceConnectivity[1][0].zones[1] == 1 && "Expecting neighbouring zone to be zone 1");

  assert(interfaceConnectivity[1][0].ownerIndex.size() == 2 && "Expected 2 owner indices");
  assert(interfaceConnectivity[1][0].neighbourIndex.size() == 2 && "Expected 2 neighbour indices");

  assert(interfaceConnectivity[1][0].ownerIndex[0][0] == 2 && "Unexpected vertex index");
  assert(interfaceConnectivity[1][0].ownerIndex[0][1] == 3 && "Unexpected vertex index");
  assert(interfaceConnectivity[1][0].ownerIndex[1][0] == 3 && "Unexpected vertex index");
  assert(interfaceConnectivity[1][0].ownerIndex[1][1] == 4 && "Unexpected vertex index");

  assert(interfaceConnectivity[1][0].neighbourIndex[0][0] == 0 && "Unexpected vertex index");
  assert(interfaceConnectivity[1][0].neighbourIndex[0][1] == 7 && "Unexpected vertex index");
  assert(interfaceConnectivity[1][0].neighbourIndex[1][0] == 7 && "Unexpected vertex index");
  assert(interfaceConnectivity[1][0].neighbourIndex[1][1] == 6 && "Unexpected vertex index");
}

void readUnstructuredMeshBoundaryConditions(ReadUnstructuredMesh &unstructuredMesh) {
  // get the boundary conditions from the structured mesh
  auto boundaryConditions = unstructuredMesh.getBoundaryConditions();

  // test that boundary conditions were read correctly
  assert(boundaryConditions.size() == 2 && "Expected 2 zones");

  assert(boundaryConditions[0].size() == 3 && "Expected 3 boundaries in zone 0");
  assert(boundaryConditions[1].size() == 3 && "Expected 3 boundaries in zone 1");

  assert(boundaryConditions[0][0].indices.size() == 2 && "Expected 2 boundary indices");
  assert(boundaryConditions[0][1].indices.size() == 2 && "Expected 2 boundary indices");
  assert(boundaryConditions[0][2].indices.size() == 2 && "Expected 2 boundary indices");

  assert(boundaryConditions[1][0].indices.size() == 2 && "Expected 2 boundary indices");
  assert(boundaryConditions[1][1].indices.size() == 2 && "Expected 2 boundary indices");
  assert(boundaryConditions[1][2].indices.size() == 2 && "Expected 2 boundary indices");

  assert(boundaryConditions[0][0].boundaryType == BC::INLET && "Expected inlet boundary condition");
  assert(boundaryConditions[0][1].boundaryType == BC::SYMMETRY && "Expected symmetry boundary condition");
  assert(boundaryConditions[0][2].boundaryType == BC::WALL && "Expected wall boundary condition");

  assert(boundaryConditions[1][0].boundaryType == BC::OUTLET && "Expected outlet boundary condition");
  assert(boundaryConditions[1][1].boundaryType == BC::SYMMETRY && "Expected symmetry boundary condition");
  assert(boundaryConditions[1][2].boundaryType == BC::WALL && "Expected wall boundary condition");

  assert(boundaryConditions[0][0].indices[0][0] == 6); assert(boundaryConditions[0][0].indices[0][1] == 7);
  assert(boundaryConditions[0][0].indices[1][0] == 7); assert(boundaryConditions[0][0].indices[1][1] == 0);

  assert(boundaryConditions[0][1].indices[0][0] == 4); assert(boundaryConditions[0][1].indices[0][1] == 5);
  assert(boundaryConditions[0][1].indices[1][0] == 5); assert(boundaryConditions[0][1].indices[1][1] == 6);

  assert(boundaryConditions[0][2].indices[0][0] == 0); assert(boundaryConditions[0][2].indices[0][1] == 1);
  assert(boundaryConditions[0][2].indices[1][0] == 1); assert(boundaryConditions[0][2].indices[1][1] == 2);

  assert(boundaryConditions[1][0].indices[0][0] == 2); assert(boundaryConditions[1][0].indices[0][1] == 3);
  assert(boundaryConditions[1][0].indices[1][0] == 3); assert(boundaryConditions[1][0].indices[1][1] == 4);

  assert(boundaryConditions[1][1].indices[0][0] == 4); assert(boundaryConditions[1][1].indices[0][1] == 5);
  assert(boundaryConditions[1][1].indices[1][0] == 5); assert(boundaryConditions[1][1].indices[1][1] == 6);

  assert(boundaryConditions[1][2].indices[0][0] == 0); assert(boundaryConditions[1][2].indices[0][1] == 1);
  assert(boundaryConditions[1][2].indices[1][0] == 1); assert(boundaryConditions[1][2].indices[1][1] == 2);
}